package gov.va.vamf.scheduling.direct.datalayer.facility;

import gov.va.vamf.scheduling.direct.domain.ClinicalService;
import gov.va.vamf.scheduling.direct.domain.Institution;
import gov.va.vamf.scheduling.direct.domain.StopCodes;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.simple.SimpleJdbcCall;
import org.springframework.stereotype.Repository;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

@Repository
public class InstitutionsDAOImpl {
    private static final String INSTITUTIONS_OUTPUT_PARAMETER_NAME = "RESULT_LIST";

    private final SimpleJdbcCall institutionsStoredProcedure;

    @Autowired
    public InstitutionsDAOImpl(
            final InstitutionsResultSetRowMapper institutionsResultSetRowMapper,
            final JdbcTemplate jdbcTemplate,
            @Value("${cdw.institutions.storedProcedure.catalog}") final String institutionsStoredProcedureCatalog,
            @Value("${cdw.institutions.storedProcedure.schema}") final String institutionsStoredProcedureSchema,
            @Value("${cdw.institutions.storedProcedure.name}") final String institutionsStoredProcedureName
    ) {
        this.institutionsStoredProcedure = new SimpleJdbcCall(jdbcTemplate)
                .withProcedureName(institutionsStoredProcedureName)
                .withCatalogName(institutionsStoredProcedureCatalog)
                .withSchemaName(institutionsStoredProcedureSchema)
                .returningResultSet(INSTITUTIONS_OUTPUT_PARAMETER_NAME, institutionsResultSetRowMapper);
    }

    public List<Institution> getInstitutions(final Collection<String> facilityCodes) {
        return getInstitutions(facilityCodes, null, null);
    }

    public List<Institution> getInstitutions(final Collection<String> facilityCodes, final String parentCode) {
        return getInstitutions(facilityCodes, parentCode, null);
    }

    public List<Institution> getInstitutions(
            final Collection<String> facilityCodes,
            final String parentCode,
            final StopCodes stopCodes
    ) {
        List<Institution> institutions = new ArrayList<>();

        String stopCodesParameter = null;
        if (stopCodes != null && !stopCodes.isEmpty()) {
            stopCodesParameter = ClinicalService.getStopCodesAsString(stopCodes);
        }

        // Unfortunately, the CDW stored procedure only takes a single station code input, so we need to make
        // multiple requests to support multiple codes.
        for (final String facilityCode : facilityCodes) {
            final MapSqlParameterSource parameters = new MapSqlParameterSource()
                    .addValue("STA3N", Integer.valueOf(facilityCode))
                    .addValue("STOP_CODE_PAIRS", stopCodesParameter);

            institutions.addAll(
                    (List<Institution>)institutionsStoredProcedure.execute(parameters).get(INSTITUTIONS_OUTPUT_PARAMETER_NAME)
            );
        }

        if (parentCode != null) {
            institutions = institutions.stream().filter(new Predicate<Institution>() { // TODO: switch to lambda
                @Override
                public boolean test(final Institution institution) {
                    return parentCode.equalsIgnoreCase(institution.getParentStationCode());
                }
            }).collect(Collectors.toList());
        }

        return institutions;
    }

    /**
     * Gets an institution by its institution code.
     * @param institutionCode the institution code to look up
     * @return an Institution matching the given institution code, or <code>null</code> if none is found
     */
    public Institution getInstitution(final String institutionCode) {

        // TODO: consider requesting a stored procedure to get an institution by 5-digit code.
        // Until then, we're forced to retrieve an institution by asking for all institutions under a parent
        // facility, then picking out the one we want. A possible memory optimization here would be to use a
        // RowMapper that returns nulls for all rows but the one we're actually looking for.

        if (institutionCode.length() < 3) {
            throw new IllegalArgumentException("Institution code must be at least three characters long.");
        }

        final String parentFacilityCode = institutionCode.substring(0, 3);

        final List<Institution> institutions = getInstitutions(Collections.singletonList(parentFacilityCode));
        for (final Institution institution : institutions) {
            if (institution.getInstitutionCode().equals(institutionCode)) {
                return institution;
            }
        }

        return null;
    }

    public Collection<Institution> getParentInstitutions(final Collection<String> rootFacilityCodes) {
        final Collection<Institution> institutionsAtRootFacilities = this.getInstitutions(rootFacilityCodes);
        final Set<Institution> parentInstitutions = institutionsAtRootFacilities
                .stream()
                .filter(new Predicate<Institution>() { // TODO: switch to lambda
                    @Override
                    public boolean test(Institution institution) {
                        return institution.isAdminParent();
                    }
                })
                .collect(Collectors.toSet());
        return parentInstitutions;
    }
}